import React from 'react';
import { useEffect, useState } from 'react';
// material
import {
  Card,
  Table,
  Stack,
  TableRow,
  TableBody,
  TableCell,
  Container,
  Typography,
  TableContainer,
} from '@mui/material';
// components
import Page from '../components/Page';
import Scrollbar from '../components/Scrollbar';
import SearchNotFound from '../components/SearchNotFound';
import Iconify from '../components/Iconify';
import {
  UcListHead,
  UcListToolbar,
  UcMoreMenu,
} from '../sections/@dashboard/utilityComponents';
//

import useDeviceId from '../hooks/useDeviceId';
import axios from 'axios';

const TABLE_HEAD = [
  { id: 'browser_name', label: 'BROWSER NAME', alignRight: false },
  { id: 'url_name', label: 'URL', alignRight: false },
  { id: 'date_time', label: 'DATE & TIME', alignRight: true },
];

import useLanguage from '../hooks/useLanguage';
import MyTablePagination from 'src/sections/@dashboard/utilityComponents/MyTablePagination';
import AlignedTableCell from 'src/sections/@dashboard/utilityComponents/AlignedTableCell';
import MuiIcon from 'src/sections/@dashboard/utilityComponents/MuiIcon';
const BrowserHistory = () => {
  const [order, setOrder] = useState('asc');
  const [selected, setSelected] = useState([]);
  const [orderBy, setOrderBy] = useState('name');
  const [filter, setFilter] = useState('');
  const [totalrows, settotalrows] = useState(0);
  const [historyList, setHistoryList] = useState([]);

  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(5);
  const { deviceID } = useDeviceId();
  const translate = useLanguage().translate;

  const isUserNotFound = historyList.length === 0;

  useEffect(async () => {
    try {
      const result = await axios.get(`browser-history/${deviceID}`, {
        params: {
          page: page + 1,
          limit: rowsPerPage,
          search: filter,
        },
      });

      console.log(result);
      setHistoryList(result.data.data.webhistory_data);
      settotalrows(result.data.data.total_count);
    } catch (error) {
      console.log(error);
    }
  }, [page, rowsPerPage, deviceID, filter]);

  const handleSelectAllClick = (event) => {
    if (event.target.checked) {
      const newSelecteds = historyList.map((n) => n.id);
      setSelected(newSelecteds);
      return;
    }
    setSelected([]);
  };

  const handleFilter = (value) => {
    setPage(0);
    setFilter(value);
  };

  //---------------------------------------

  //SortAllFuntionsForToolbar
  const sortAll = () => {
    let sortId = TABLE_HEAD[1].id;
    const isAsc = orderBy === sortId && order === 'asc';
    setOrder(isAsc ? 'desc' : 'asc');
    setOrderBy(sortId);
  };

  const handleRequestSort = (event, property) => {
    const isAsc = orderBy === property && order === 'asc';
    setOrder(isAsc ? 'desc' : 'asc');
    setOrderBy(property);
  };

  const filterhistoryList = applySortFilter(
    historyList,
    getComparator(order, orderBy),
    filter
  );

  function applySortFilter(array, comparator, query) {
    const stabilizedThis = array.map((el, index) => [el, index]);
    stabilizedThis.sort((a, b) => {
      const order = comparator(a[0], b[0]);
      if (order !== 0) return order;
      return a[1] - b[1];
    });
    return stabilizedThis.map((el) => el[0]);
  }

  function getComparator(order, orderBy) {
    return order === 'desc'
      ? (a, b) => descendingComparator(a, b, orderBy)
      : (a, b) => -descendingComparator(a, b, orderBy);
  }

  function descendingComparator(a, b, orderBy) {
    if (b[orderBy] < a[orderBy]) {
      return -1;
    }
    if (b[orderBy] > a[orderBy]) {
      return 1;
    }
    return 0;
  }

  return (
    <Page title={translate('Browser History')}>
      <Container>
        <Stack
          direction='row'
          alignItems='center'
          justifyContent='space-between'
          mb={5}
        >
          <Typography variant='h4' gutterBottom>
            {translate('Browser History')}
          </Typography>
        </Stack>

        <Card>
          <UcListToolbar
            onSort={sortAll}
            placeholder='Search By Url...'
            numSelected={selected.length}
            onFilter={handleFilter}
          />
          <Scrollbar>
            <TableContainer sx={{ minWidth: 800 }}>
              <Table>
                <UcListHead
                  order={order}
                  orderBy={orderBy}
                  headLabel={TABLE_HEAD}
                  rowCount={historyList.length}
                  numSelected={selected.length}
                  onRequestSort={handleRequestSort}
                  onSelectAllClick={handleSelectAllClick}
                />
                <TableBody>
                  {filterhistoryList.map((row) => {
                    const { id, browser_name, url_name, date_time, action } =
                      row;
                    const isItemSelected = selected.indexOf(id) !== -1;

                    return (
                      <TableRow
                        hover
                        key={id}
                        tabIndex={-1}
                        role='checkbox'
                        selected={isItemSelected}
                        aria-checked={isItemSelected}
                      >
                        <TableCell component='th' scope='row' padding='normal'>
                          <Stack
                            direction='row'
                            alignItems='center'
                            spacing={2}
                          >
                            <Typography variant='subtitle2' noWrap>
                              <MuiIcon icon='arcticons:browser' />

                              {browser_name}
                            </Typography>
                          </Stack>
                        </TableCell>

                        <AlignedTableCell flow='left'>
                          <Typography variant='subtitle2' noWrap>
                            <MuiIcon icon='arcticons:url-forwarder' />

                            {url_name}
                          </Typography>
                        </AlignedTableCell>
                        <AlignedTableCell flow='right'>
                          <Typography variant='subtitle2' noWrap>
                            <MuiIcon icon='system-uicons:calendar-date' />

                            {date_time}
                          </Typography>
                        </AlignedTableCell>
                      </TableRow>
                    );
                  })}
                </TableBody>
                {isUserNotFound && (
                  <TableBody>
                    <TableRow>
                      <TableCell align='center' colSpan={6} sx={{ py: 3 }}>
                        <SearchNotFound searchQuery={filter} />
                      </TableCell>
                    </TableRow>
                  </TableBody>
                )}
              </Table>
            </TableContainer>
          </Scrollbar>

          <MyTablePagination
            count={totalrows}
            rowsPerPage={rowsPerPage}
            page={page}
            setPage={setPage}
            setRowsPerPage={setRowsPerPage}
          />
        </Card>
      </Container>
    </Page>
  );
};

export default BrowserHistory;
