import React, { useEffect, useState } from 'react';
// material
import {
  Card,
  Table,
  Stack,
  Checkbox,
  TableRow,
  TableBody,
  TableCell,
  Container,
  Typography,
  TableContainer,
} from '@mui/material';

import SearchNotFound from '../components/SearchNotFound';
import Scrollbar from '../components/Scrollbar';
import useDeviceId from '../hooks/useDeviceId';

import {
  UcListHead,
  UcListToolbar,
  UcMoreMenu,
} from '../sections/@dashboard/utilityComponents';

// components
import Page from '../components/Page';
import axios from 'axios';
import Iconify from 'src/components/Iconify';

import useLanguage from '../hooks/useLanguage';
import MyTablePagination from 'src/sections/@dashboard/utilityComponents/MyTablePagination';
import AlignedTableCell from 'src/sections/@dashboard/utilityComponents/AlignedTableCell';
import MuiIcon from 'src/sections/@dashboard/utilityComponents/MuiIcon';

const TABLE_HEAD = [

  { id: 'id', label: 'ID', alignRight: false },
  { id: 'title', label: 'TITLE', alignRight: false },
  { id: 'start_time', label: 'START TIME', alignRight: false },
  { id: 'finish_time', label: 'FINISH TIME', alignRight: false },
  { id: 'event_location', label: 'EVENT LOCATION', alignRight: false },
];

const CalenderEventsPage = () => {
  const [selected, setSelected] = useState([]);
  const [locations, setLocations] = useState([]);
  const { deviceID } = useDeviceId();
  const translate = useLanguage().translate;
  const [filter, setFilter] = useState('');
  const [page, setPage] = useState(0);
  const [totalrows, settotalrows] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(5);

  const [order, setOrder] = useState('asc');
  const [orderBy, setOrderBy] = useState('title');

  const isUserNotFound = locations.length === 0;

  useEffect(async () => {
    try {
      const result = await axios.get(`calendar_event/${deviceID}`, {
        params: {
          page: page + 1,
          limit: rowsPerPage,
          search: filter,
        },
      });

      setLocations(result.data.data.calendar_enent);
      settotalrows(result.data.data.total_count);
    } catch (error) {
      console.log(error);
    }
  }, [page, rowsPerPage, filter, deviceID]);

  const handleSelectAllClick = (event) => {
    if (event.target.checked) {
      const newSelecteds = locations.map((n) => n.id);
      setSelected(newSelecteds);
      return;
    }
    setSelected([]);
  };

  const filterList = (value) => {
    setPage(0);
    setFilter(value);
  };
  //------------------------------------

  //SortAllFuntionsForToolbar
  const sortAll = () => {
    let sortId = TABLE_HEAD[0].id;
    const isAsc = orderBy === sortId && order === 'asc';
    setOrder(isAsc ? 'desc' : 'asc');
    setOrderBy(sortId);
  };
  const handleRequestSort = (event, property) => {
    const isAsc = orderBy === property && order === 'asc';
    setOrder(isAsc ? 'desc' : 'asc');
    setOrderBy(property);
  };

  const filterlocations = applySortFilter(
    locations,
    getComparator(order, orderBy),
    filter
  );

  function applySortFilter(array, comparator, query) {
    const stabilizedThis = array.map((el, index) => [el, index]);
    stabilizedThis.sort((a, b) => {
      const order = comparator(a[0], b[0]);
      if (order !== 0) return order;
      return a[1] - b[1];
    });
    return stabilizedThis.map((el) => el[0]);
  }

  function getComparator(order, orderBy) {
    return order === 'desc'
      ? (a, b) => descendingComparator(a, b, orderBy)
      : (a, b) => -descendingComparator(a, b, orderBy);
  }

  function descendingComparator(a, b, orderBy) {
    if (b[orderBy] < a[orderBy]) {
      return -1;
    }
    if (b[orderBy] > a[orderBy]) {
      return 1;
    }
    return 0;
  }
  return (
    <Page title={translate('Calender Events')}>
      <Container>
        <Stack
          direction='row'
          alignItems='center'
          justifyContent='space-between'
          mb={5}
        >
          <Typography variant='h4' gutterBottom>
            {translate('Calender Events')}
          </Typography>
        </Stack>

        <Card>
          <UcListToolbar
            onSort={sortAll}
            placeholder='Search By Title...'
            numSelected={selected.length}
            onFilter={filterList}
          />
          <Scrollbar>
            <>
              <TableContainer sx={{ minWidth: 800 }}>
                <Table>
                  <UcListHead
                    order={order}
                    orderBy={orderBy}
                    headLabel={TABLE_HEAD}
                    rowCount={locations.length}
                    numSelected={selected.length}
                    onRequestSort={handleRequestSort}
                    onSelectAllClick={handleSelectAllClick}
                  />
                  <TableBody>
                    {filterlocations.map((row) => {
                      const {
                        id,
                        title,
                        start_time,
                        finish_time,
                        event_location,
                        action,
                      } = row;
                      const isItemSelected = selected.indexOf(id) !== -1;

                      return (
                        <TableRow
                          hover
                          key={id}
                          tabIndex={-1}
                          role='checkbox'
                          selected={isItemSelected}
                          aria-checked={isItemSelected}
                        >
                          <AlignedTableCell flow='left'>
                            {id}
                          </AlignedTableCell>

                          <AlignedTableCell flow='left'>
                            {title}
                          </AlignedTableCell>

                          <AlignedTableCell flow='left'>
                            {start_time}
                          </AlignedTableCell>
                          <AlignedTableCell flow='left'>
                            {finish_time}
                          </AlignedTableCell>
                          <AlignedTableCell flow='left'>
                            <Typography variant='subtitle2' noWrap>

                              <MuiIcon icon='akar-icons:location' />
                              {event_location}
                            </Typography>
                          </AlignedTableCell>
                        </TableRow>
                      );
                    })}
                  </TableBody>
                  {isUserNotFound && (
                    <TableBody>
                      <TableRow>
                        <TableCell align='center' colSpan={6} sx={{ py: 3 }}>
                          <SearchNotFound searchQuery={filter} />
                        </TableCell>
                      </TableRow>
                    </TableBody>
                  )}
                </Table>
              </TableContainer>
            </>
          </Scrollbar>

          <MyTablePagination
            count={totalrows}
            rowsPerPage={rowsPerPage}
            page={page}
            setPage={setPage}
            setRowsPerPage={setRowsPerPage}
          />
        </Card>
      </Container>
    </Page>
  );
};

export default CalenderEventsPage;
